import socket, struct, time

# Set the address of the Cerebot board
dst_addr = '00183E00D7EB'.decode('hex')
# Open a socket on the eth0
sock = socket.socket(socket.AF_PACKET, socket.SOCK_RAW)
sock.bind(("eth0", 0x1))
# Acquire MAC address of local machine
if_name, if_proto, pkt_type, hw_type, hw_addr = sock.getsockname()

# Create and initialize the frame buffer
frame_buffer_size = 1536
frame_buffer = [0] * frame_buffer_size

def cube_init():
    '''Sets the cube into ethernet mode.'''
    # Generate and send the frame
    txFrame = struct.pack("!6s6sH", dst_addr, hw_addr, 1)
    txOpcode = "01".decode('hex')
    sock.send(txFrame + txOpcode)
    # Wait a few seconds for the cube to reset itself
    time.sleep(6)

def cube_reset():
    '''Resets the cube into idle mode.'''
    # Generate and send the frame
    txFrame = struct.pack("!6s6sH", dst_addr, hw_addr, 1)
    txOpcode = "02".decode('hex')
    sock.send(txFrame + txOpcode)

def cube_clear():
    '''Clear the cube's internal buffer.'''
    # Generate and send the frame
    txFrame = struct.pack("!6s6sH", dst_addr, hw_addr, 1)
    txOpcode = "0A".decode('hex')
    sock.send(txFrame + txOpcode)

def cube_brightness(value):
    '''Sets the global brightness value from 0 to 255.'''
    # Generate and send the frame
    txFrame = struct.pack("!6s6sH", dst_addr, hw_addr, 2)
    txOpcode = "0B".decode('hex')
    txData = format(value, '02x').decode('hex')
    sock.send(txFrame + txOpcode + txData)

def cube_rotate(direction):
    '''Rotate the entire cube (0 = clockwise, 1 = counterclockwise).'''
    # Generate and send the frame
    txFrame = struct.pack("!6s6sH", dst_addr, hw_addr, 2)
    txOpcode = "0C".decode('hex')
    txData = format(direction, '02x').decode('hex')
    sock.send(txFrame + txOpcode + txData)

def cube_rotate_shell(direction, shell):
    '''Rotate a specific layer (0 = clockwise, 0 = outermost layer).'''
    # Generate and send the frame
    txFrame = struct.pack("!6s6sH", dst_addr, hw_addr, 3)
    txOpcode = "0D".decode('hex')
    txData = ''.join("%02x%02x" % (direction, shell))
    sock.send(txFrame + txOpcode + txData.decode('hex'))

# def cube_update():
#     '''Update the cube with the current frame buffer.
#         Note: this requires jumbo frames (1536 bytes).'''
#     # Generate the header, opcode, and format the frame buffer
#     txFrame = struct.pack("!6s6sH", dst_addr, hw_addr, 0x0601)
#     txOpcode = "10".decode('hex')
#     txData = ''.join("%02x" % (x) for x in frame_buffer)
#     sock.send(txFrame + txOpcode + txData.decode('hex'))

def cube_update_pixel(x, y, z, r, g, b):
    '''Set a specific pixel on the cube.'''
    txFrame = struct.pack("!6s6sH", dst_addr, hw_addr, 7)
    txOpcode = "11".decode('hex')
    frame = [x, y, z, r, g, b]
    txData = ''.join("%02x" % (x) for x in frame)
    sock.send(txFrame + txOpcode + txData.decode('hex'))

def cube_update():
    '''Updates the cube with the current frame buffer.
        The buffer is sent in three frames, one for each color channel.'''
    txFrame = struct.pack("!6s6sH", dst_addr, hw_addr, 514)
    txOpcode = "12".decode('hex')
    for c in range(3):
        txColorCh = "%02x" % c
        txData = ''.join( ["%02x" % (x) for x in frame_buffer[c::3]])
        payload = txFrame + txOpcode + txColorCh.decode('hex') + txData.decode('hex')
        sock.send(payload)
        time.sleep(0.01)    # This value can be smaller (and should be) to reduce flickering

def cube_update_text_scrolling(string, r, g, b, update_rate):
    '''Sets the scrolling text on the cube.'''
    txFrame = struct.pack("!6s6sH", dst_addr, hw_addr, len(string) + 6)
    txOpcode = "20".decode('hex')
    txData = ''.join("%02x%02x%02x%02x%02x" % (len(string), r, g, b, update_rate))
    txString = string.encode('hex')
    sock.send(txFrame + txOpcode + txData.decode('hex') + txString.decode('hex'))

def cube_update_text_static(string, r, g, b):
    '''Sets the static text on the cube.'''
    txFrame = struct.pack("!6s6sH", dst_addr, hw_addr, len(string) + 5)
    txOpcode = "21".decode('hex')
    txData = ''.join("%02x%02x%02x%02x" % (len(string), r, g, b))
    txString = string.encode('hex')
    sock.send(txFrame + txOpcode + txData.decode('hex') + txString.decode('hex'))

def cube_update_text_insert(character, r, g, b, delay):
    '''Appends a character to the beginning of the text buffer.'''
    txFrame = struct.pack("!6s6sH", dst_addr, hw_addr, 6)
    txOpcode = "22".decode('hex')
    txData = ''.join("%02x%02x%02x%02x%02x" % (r, g, b, delay, character))
    sock.send(txFrame + txOpcode + txData.decode('hex'))

def cube_update_waterfall(c0, c1, c2, c3, c4, c5, c6, c7):
    '''Fills in one row and shifts rows by one.'''
    txFrame = struct.pack("!6s6sH", dst_addr, hw_addr, 9)
    txOpcode = "30".decode('hex')
    txData = ''.join("%02x%02x%02x%02x%02x%02x%02x%02x" % (c0, c1, c2, c3, c4, c5, c6, c7))
    sock.send(txFrame + txOpcode + txData.decode('hex'))

def cube_update_sphere(layer, r, g, b):
    '''Sets the sphere layer to the specified color.'''
    txFrame = struct.pack("!6s6sH", dst_addr, hw_addr, 5)
    txOpcode = "31".decode('hex')
    txData = ''.join("%02x%02x%02x%02x" % (layer, r, g, b))
    sock.send(txFrame + txOpcode + txData.decode('hex'))

def fb_clear():
    '''Clears the frame buffer/'''
    for row in range(8):
    	for column in range(8):
    	    for layer in range(8):
    		  fb_set_pixel(row, column, layer, 0x00, 0x00, 0x00)

def fb_set_pixel(row, column, layer, R, G, B):
    '''Sets a pixel to the given color in the frame buffer/'''
    frame_buffer[layer * 192 + column * 24 + row * 3 + 0] = R
    frame_buffer[layer * 192 + column * 24 + row * 3 + 1] = G
    frame_buffer[layer * 192 + column * 24 + row * 3 + 2] = B

def fb_set_layer(layer, R, G, B):
    '''Sets an entire layer to the given color in the frame buffer'''
    for row in range(8):
	for column in range(8):
	    fb_set_pixel(row, column, layer, R, G, B)
    
def fb_set_all(R, G, B):
    '''Sets all the pixels in the frame buffer to the given color'''
    for row in range(8):
	for column in range(8):
	    for layer in range(8):
		fb_set_pixel(row, column, layer, R, G, B)
